#ifndef AHLGREN_MODES
#define AHLGREN_MODES

#include <vector>
#include <tuple>

#include "functor.h"
#include "parameters.h"


namespace lp
{
	using namespace std;

	// One mode
	class Mode {
	public:
		enum marker { normal, input, output, ground };
		typedef Functor::position position;
		typedef Functor::arity_type arity_type;
		struct pmarker
		{
			pmarker() {}
			pmarker(const position& p, marker m, id_type s) : pos(p), iotype(m), type(s) {}
			pmarker(const pmarker& p) : pos(p.pos), iotype(p.iotype), type(p.type) {}
			pmarker& operator=(const pmarker& p) { 
				if (this != &p) { pos = p.pos; iotype = p.iotype; type = p.type; } 
				return *this; 
			}
			// Data members
			position pos;
			marker iotype;
			id_type type;
		};
		typedef vector<pmarker> pmarker_vec;
		typedef pmarker_vec::iterator iterator;
		typedef pmarker_vec::const_iterator const_iterator;
		enum prop { head, rec, maxc, functional, reflexive, symmetric, transitive, idempotent, SIZE };

		Mode();
		Mode(const Functor& f, bool h, int r = numeric_limits<int>::max());
		Mode(Functor&& f, bool h, int r = numeric_limits<int>::max());
		// Copy constructor/assignment
		//Mode(const Mode& m);
		//Mode& operator=(const Mode&);
		// Move constructor/assignment
		Mode(Mode&& m);
		Mode& operator=(Mode&& m);
		// Compare
		bool operator==(const Mode& m) const;
		bool operator!=(const Mode& m) const { return !(*this == m); }
		bool operator<(const Mode& m) const;
		// Get info
		bool is_head() const { return props[prop::head] != 0; }
		int recall() const { return props[prop::rec]; }
		int max_occurs() const { return props[prop::maxc]; }
		const Functor& atom() const { return _atom; }

		// Is mode set?
		operator bool() const { return !_atom.is_nil(); }

		static marker get_marker(const Functor& f);

		// Get prototype instance
		Functor prototype() const;

		// Get generative clause
		//Clause generating_clause() const;

		// Get all variables of place marker type pm in f
		void variables(marker pm, const Functor& f, set<id_type>&) const;

		// Iterate through all place markers
		const_iterator begin() const { return placemarkers.begin(); }
		const_iterator end() const { return placemarkers.end(); }
		// Iterate through inputs/outputs/ground terms
		template<Mode::marker MTYPE> class selective_iterator;
		typedef selective_iterator<Mode::input> input_iterator;
		typedef selective_iterator<Mode::output> output_iterator;
		typedef selective_iterator<Mode::ground> ground_iterator;
		input_iterator input_begin() const;
		input_iterator input_end() const;
		output_iterator output_begin() const;
		output_iterator output_end() const;
		ground_iterator ground_begin() const;
		ground_iterator ground_end() const;

		// Print mode (as clause)
		void print(ostream&) const;
	public:
		std::array<int,int(prop::SIZE)> props;
	protected:
		Functor _atom;
		// store position of all place markers and their type
		pmarker_vec placemarkers;

		// Initialize place markers from _atom
		void initpm();
	};

	template <Mode::marker MTYPE>
	class Mode::selective_iterator {
	public:
		typedef Mode::const_iterator iter;
		typedef forward_iterator_tag iterator_category;
		typedef Mode::pmarker value_type;
		typedef const value_type* pointer;
		typedef const value_type& reference;
		typedef ptrdiff_t difference_type;
		selective_iterator(iter b, iter e) : i(b), end(e) {
			// Fast forward to first match
			while (i != end && i->iotype != MTYPE) ++i;
		}
		selective_iterator operator++() { 
			do { ++i; } while (i != end && i->iotype != MTYPE);
			return *this;
		}
		selective_iterator operator++(int) { selective_iterator it(*this); ++(*this); return it; }
		bool operator==(const selective_iterator& it) const { return i == it.i && end == it.end; }
		bool operator!=(const selective_iterator& it) const { return !(*this == it); }
		reference operator*() const { return *i; }
		pointer operator->() const { return &(*i); }
	protected:
		iter i, end;
	};


	//====================== Global Functions ======================//

	inline Mode::input_iterator Mode::input_begin() const { return input_iterator(begin(),end()); }
	inline Mode::input_iterator Mode::input_end() const { return input_iterator(end(),end()); }
	inline Mode::output_iterator Mode::output_begin() const { return output_iterator(begin(),end()); }
	inline Mode::output_iterator Mode::output_end() const { return output_iterator(end(),end()); }
	inline Mode::ground_iterator Mode::ground_begin() const { return ground_iterator(begin(),end()); }
	inline Mode::ground_iterator Mode::ground_end() const { return ground_iterator(end(),end()); }

	inline ostream& operator<<(ostream& os, const Mode& m) { m.print(os); return os; }

	// Create Input/Output map from clause and modes
	typedef std::vector<std::pair<std::set<id_type>,set<id_type>>> io_map;
	io_map make_io_map(const Functor& f, const vector<Mode>& modes);
	// Test Mode I/O conformance
	bool is_chained(const Functor& f, const vector<Mode>&);
	bool is_returned(const Functor& f, const vector<Mode>&);
	bool is_argument(const Functor& f, const vector<Mode>&);
	// TODO: is_functional? (predicate/literal specific)
	//bool is_mode_conformant(const Functor& cand, const parameters& params, const vector<Mode>& modes);
	bool is_mode_conformant(const Functor& cand, const parameters& params, const io_map& iomap);

	inline Mode::Mode()
	{
		props.fill(0);
		props[prop::head] = 1;
		props[prop::rec] = std::numeric_limits<int>::max();
		props[prop::maxc] = std::numeric_limits<int>::max();
	
	}

	inline Mode::Mode(const Functor& f, bool h, int r) : _atom(f)
	{ 
		props.fill(0);
		props[prop::head] = h;
		props[prop::rec] = r;
		props[prop::maxc] = std::numeric_limits<int>::max();
		initpm(); 
	}

	inline Mode::Mode(Functor&& f, bool h, int r) : _atom(std::move(f))
	{ 
		props.fill(0);
		props[prop::head] = h;
		props[prop::rec] = r;
		props[prop::maxc] = std::numeric_limits<int>::max();
		initpm(); 
	}

	inline Mode::Mode(Mode&& m) 
		: props(std::move(m.props)), _atom(std::move(m._atom)), placemarkers(std::move(m.placemarkers)) { }


	inline bool Mode::operator==(const Mode& m) const
	{
		return props == m.props && atom() == m.atom();
	}

	inline Mode::marker Mode::get_marker(const Functor& f)
	{
		if (f.arity() != 1 || !f.arg_first()->is_leaf()) return Mode::normal;
		if (f.id() == plus_id) return Mode::input;
		if (f.id() == minus_id) return Mode::output;
		if (f.id() == hash_id) return Mode::ground;
		return normal;
	}

	inline void Mode::variables(marker pm, const Functor& f, set<id_type>& vars) const
	{
		for (const Functor& n : atom()) {
			if (this->get_marker(n) == pm) { 
				vars.insert( f.get(Functor::position(&n))->id() );
			}
		}
	}




}

#endif

